// Z80 (Zed Eight-Ty) Interface
#include "burnint.h"
#include "zet2.h"

#define MAX_Z80		8
static struct Zet2Ext * Zet2CPUContext[MAX_Z80] = { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };
 
typedef UINT8 (__fastcall *pZet2InHandler)(UINT16 a);
typedef void (__fastcall *pZet2OutHandler)(UINT16 a, UINT8 d);
typedef UINT8 (__fastcall *pZet2ReadHandler)(UINT16 a);
typedef void (__fastcall *pZet2WriteHandler)(UINT16 a, UINT8 d);
 
struct Zet2Ext {
	Z80_Regs reg;
	
	UINT8* pZet2MemMap[0x100 * 4];

	pZet2InHandler Zet2In;
	pZet2OutHandler Zet2Out;
	pZet2ReadHandler Zet2Read;
	pZet2WriteHandler Zet2Write;
	
	UINT8 BusReq;
};
 
static INT32 nZet2CyclesDone[MAX_Z80];
static INT32 nZet2CyclesTotal;
static INT32 nZ80ICount[MAX_Z80];
static UINT32 Z80EA[MAX_Z80];

static INT32 nOpenedCPU = -1;
static INT32 nCPUCount = 0;
INT32 nHasZet2 = -1;

UINT8 __fastcall Zet2DummyReadHandler(UINT16) { return 0; }
void __fastcall Zet2DummyWriteHandler(UINT16, UINT8) { }
UINT8 __fastcall Zet2DummyInHandler(UINT16) { return 0; }
void __fastcall Zet2DummyOutHandler(UINT16, UINT8) { }

UINT8 __fastcall Zet2ReadIO(UINT32 a)
{
	return Zet2CPUContext[nOpenedCPU]->Zet2In(a);
}

void __fastcall Zet2WriteIO(UINT32 a, UINT8 d)
{
	Zet2CPUContext[nOpenedCPU]->Zet2Out(a, d);
}

UINT8 __fastcall Zet2ReadProg(UINT32 a)
{
	// check mem map
	UINT8 * pr = Zet2CPUContext[nOpenedCPU]->pZet2MemMap[0x000 | (a >> 8)];
	if (pr != NULL) {
		return pr[a & 0xff];
	}
	
	// check handler
	if (Zet2CPUContext[nOpenedCPU]->Zet2Read != NULL) {
		return Zet2CPUContext[nOpenedCPU]->Zet2Read(a);
	}
	
	return 0;
}

void __fastcall Zet2WriteProg(UINT32 a, UINT8 d)
{
	// check mem map
	UINT8 * pr = Zet2CPUContext[nOpenedCPU]->pZet2MemMap[0x100 | (a >> 8)];
	if (pr != NULL) {
		pr[a & 0xff] = d;
		return;
	}
	
	// check handler
	if (Zet2CPUContext[nOpenedCPU]->Zet2Write != NULL) {
		Zet2CPUContext[nOpenedCPU]->Zet2Write(a, d);
		return;
	}
}

UINT8 __fastcall Zet2ReadOp(UINT32 a)
{
	// check mem map
	UINT8 * pr = Zet2CPUContext[nOpenedCPU]->pZet2MemMap[0x200 | (a >> 8)];
	if (pr != NULL) {
		return pr[a & 0xff];
	}
	
	// check read handler
	if (Zet2CPUContext[nOpenedCPU]->Zet2Read != NULL) {
		return Zet2CPUContext[nOpenedCPU]->Zet2Read(a);
	}
	
	return 0;
}

UINT8 __fastcall Zet2ReadOpArg(UINT32 a)
{
	// check mem map
	UINT8 * pr = Zet2CPUContext[nOpenedCPU]->pZet2MemMap[0x300 | (a >> 8)];
	if (pr != NULL) {
		return pr[a & 0xff];
	}
	
	// check read handler
	if (Zet2CPUContext[nOpenedCPU]->Zet2Read != NULL) {
		return Zet2CPUContext[nOpenedCPU]->Zet2Read(a);
	}
	
	return 0;
}

void Zet2SetReadHandler(UINT8 (__fastcall *pHandler)(UINT16))
{
	Zet2CPUContext[nOpenedCPU]->Zet2Read = pHandler;
}

void Zet2SetWriteHandler(void (__fastcall *pHandler)(UINT16, UINT8))
{

	Zet2CPUContext[nOpenedCPU]->Zet2Write = pHandler;
}

void Zet2SetInHandler(UINT8 (__fastcall *pHandler)(UINT16))
{
	Zet2CPUContext[nOpenedCPU]->Zet2In = pHandler;
}

void Zet2SetOutHandler(void (__fastcall *pHandler)(UINT16, UINT8))
{
	Zet2CPUContext[nOpenedCPU]->Zet2Out = pHandler;
}

void Zet2NewFrame()
{
	for (INT32 i = 0; i < nCPUCount; i++) {
		nZet2CyclesDone[i] = 0;
	}
	nZet2CyclesTotal = 0;
}

static void Zet2CheatWriteROM(UINT32 a, UINT8 d)
{
	ZetWriteRom(a, d);
}

static UINT8 Zet2CheatRead(UINT32 a)
{
	return Zet2ReadByte(a);
}

/*static cpu_core_config Zet2CheatCpuConfig =
{
	Zet2Open,
	Zet2Close,
	Zet2CheatRead,
	Zet2CheatWriteROM,
	Zet2GetActive,
	Zet2TotalCycles,
	Zet2NewFrame,
	Zet2Run,
	Zet2RunEnd,
	Zet2Reset,
	(1<<16),	// 0x10000
	0
};*/

INT32 Zet2Init(INT32 nCPU)
{
	//DebugCPU_ZetInitted = 1;

	nOpenedCPU = -1;

	Zet2CPUContext[nCPU] = (struct Zet2Ext*)BurnMalloc(sizeof(Zet2Ext));
	memset (Zet2CPUContext[nCPU], 0, sizeof(Zet2Ext));

	if (nCPU == 0) { // not safe!
		Z80Init();
	}

	{
		Zet2CPUContext[nCPU]->Zet2In = Zet2DummyInHandler;
		Zet2CPUContext[nCPU]->Zet2Out = Zet2DummyOutHandler;
		Zet2CPUContext[nCPU]->Zet2Read = Zet2DummyReadHandler;
		Zet2CPUContext[nCPU]->Zet2Write = Zet2DummyWriteHandler;
		Zet2CPUContext[nCPU]->BusReq = 0;
		// TODO: Z80Init() will set IX IY F regs with default value, so get them ...
		Z80GetContext(&Zet2CPUContext[nCPU]->reg);
		
		nZet2CyclesDone[nCPU] = 0;
		nZ80ICount[nCPU] = 0;
		
		for (INT32 j = 0; j < (0x0100 * 4); j++) {
			Zet2CPUContext[nCPU]->pZet2MemMap[j] = NULL;
		}
	}

	nZet2CyclesTotal = 0;

	Z80SetIOReadHandler(Zet2ReadIO);
	Z80SetIOWriteHandler(Zet2WriteIO);
	Z80SetProgramReadHandler(Zet2ReadProg);
	Z80SetProgramWriteHandler(Zet2WriteProg);
	Z80SetCPUOpReadHandler(Zet2ReadOp);
	Z80SetCPUOpArgReadHandler(Zet2ReadOpArg);
	
	nCPUCount = (nCPU+1) % MAX_Z80;

	nHasZet2 = nCPU+1;

//	CpuCheatRegister(nCPU, &Zet2CheatCpuConfig);

	return 0;
}

#if 0
INT32 Zet2Init(INT32 nCount)
{
	//DebugCPU_ZetInitted = 1;

	nOpenedCPU = -1;
	
	Zet2CPUContext = (struct Zet2Ext *) malloc(nCount * sizeof(Zet2Ext));
	if (Zet2CPUContext == NULL) return 1;
	memset(Zet2CPUContext, 0, nCount * sizeof(Zet2Ext));
	
	Z80Init();
	
	for (INT32 i = 0; i < nCount; i++) {
		Zet2CPUContext[i].Zet2In = Zet2DummyInHandler;
		Zet2CPUContext[i].Zet2Out = Zet2DummyOutHandler;
		Zet2CPUContext[i].Zet2Read = Zet2DummyReadHandler;
		Zet2CPUContext[i].Zet2Write = Zet2DummyWriteHandler;
		Zet2CPUContext[i].BusReq = 0;
		// TODO: Z80Init() will set IX IY F regs with default value, so get them ...
		Z80GetContext(&Zet2CPUContext[i].reg);
		
		nZet2CyclesDone[i] = 0;
		nZ80ICount[i] = 0;
		
		for (INT32 j = 0; j < (0x0100 * 4); j++) {
			Zet2CPUContext[i].pZet2MemMap[j] = NULL;
		}
	}
	
	nZet2CyclesTotal = 0;
	
	Z80SetIOReadHandler(Zet2ReadIO);
	Z80SetIOWriteHandler(Zet2WriteIO);
	Z80SetProgramReadHandler(Zet2ReadProg);
	Z80SetProgramWriteHandler(Zet2WriteProg);
	Z80SetCPUOpReadHandler(Zet2ReadOp);
	Z80SetCPUOpArgReadHandler(Zet2ReadOpArg);
	
	nCPUCount = nCount % MAX_Z80;

	nHasZet2 = nCount;

	for (INT32 i = 0; i < nCount; i++)
		CpuCheatRegister(0x0004, i);

	return 0;
}
#endif

UINT8 Zet2ReadByte(UINT16 address)
{
	if (nOpenedCPU < 0) return 0;

	return Zet2ReadProg(address);
}

void Zet2WriteByte(UINT16 address, UINT8 data)
{
	if (nOpenedCPU < 0) return;

	Zet2WriteProg(address, data);
}

void Zet2WriteRom(UINT16 address, UINT8 data)
{

	if (nOpenedCPU < 0) return;

	if (Zet2CPUContext[nOpenedCPU]->pZet2MemMap[0x200 | (address >> 8)] != NULL) {
		Zet2CPUContext[nOpenedCPU]->pZet2MemMap[0x200 | (address >> 8)][address & 0xff] = data;
	}
	
	if (Zet2CPUContext[nOpenedCPU]->pZet2MemMap[0x300 | (address >> 8)] != NULL) {
		Zet2CPUContext[nOpenedCPU]->pZet2MemMap[0x300 | (address >> 8)][address & 0xff] = data;
	}
	
	Zet2WriteProg(address, data);
}

void Zet2Close()
{

	Z80GetContext(&Zet2CPUContext[nOpenedCPU]->reg);
	nZet2CyclesDone[nOpenedCPU] = nZet2CyclesTotal;
	nZ80ICount[nOpenedCPU] = z80_ICount;
	Z80EA[nOpenedCPU] = EA;

	nOpenedCPU = -1;
}

void Zet2Open(INT32 nCPU)
{

	Z80SetContext(&Zet2CPUContext[nCPU]->reg);
	nZet2CyclesTotal = nZet2CyclesDone[nCPU];
	z80_ICount = nZ80ICount[nCPU];
	EA = Z80EA[nCPU];

	nOpenedCPU = nCPU;
}

INT32 Zet2GetActive()
{

	return nOpenedCPU;
}

INT32 Zet2Run(INT32 nCycles)
{

	if (nCycles <= 0) return 0;
	
	if (Zet2CPUContext[nOpenedCPU]->BusReq) {
		nZet2CyclesTotal += nCycles;
		return nCycles;
	}
	
	nCycles = Z80Execute(nCycles);
	
	nZet2CyclesTotal += nCycles;
	
	return nCycles;
}

void Zet2RunAdjust(INT32 /*nCycles*/)
{

}

void Zet2RunEnd()
{
	//z80_ICount = 0;
}

// This function will make an area callback ZetRead/ZetWrite
INT32 Zet2MemCallback(INT32 nStart, INT32 nEnd, INT32 nMode)
{

	UINT8 cStart = (nStart >> 8);
	UINT8 **pMemMap = Zet2CPUContext[nOpenedCPU]->pZet2MemMap;

	for (UINT16 i = cStart; i <= (nEnd >> 8); i++) {
		switch (nMode) {
			case 0:
				pMemMap[0     + i] = NULL;
				break;
			case 1:
				pMemMap[0x100 + i] = NULL;
				break;
			case 2:
				pMemMap[0x200 + i] = NULL;
				pMemMap[0x300 + i] = NULL;
				break;
		}
	}

	return 0;
}

void Zet2Exit()
{
	Z80Exit();

	for (INT32 i = 0; i < MAX_Z80; i++) {
		if (Zet2CPUContext[i]) {
			BurnFree (Zet2CPUContext[i]);
		}
	}

	nCPUCount = 0;
	nHasZet2 = -1;
	
//	DebugCPU_ZetInitted = 0;
}


// This function will make an area callback ZetRead/ZetWrite
INT32 Zet2UnmapMemory(INT32 nStart, INT32 nEnd, INT32 nFlags)
{

	UINT8 cStart = (nStart >> 8);
	UINT8 **pMemMap = Zet2CPUContext[nOpenedCPU]->pZet2MemMap;

	for (UINT16 i = cStart; i <= (nEnd >> 8); i++) {
		if (nFlags & (1 << 0)) pMemMap[0     + i] = NULL; // READ
		if (nFlags & (1 << 1)) pMemMap[0x100 + i] = NULL; // WRITE
		if (nFlags & (1 << 2)) pMemMap[0x200 + i] = NULL; // OP
		if (nFlags & (1 << 3)) pMemMap[0x300 + i] = NULL; // ARG
	}

	return 0;
}

void Zet2MapMemory(UINT8 *Mem, INT32 nStart, INT32 nEnd, INT32 nFlags)
{

	UINT8 cStart = (nStart >> 8);
	UINT8 **pMemMap = Zet2CPUContext[nOpenedCPU]->pZet2MemMap;

	for (UINT16 i = cStart; i <= (nEnd >> 8); i++) {
		if (nFlags & (1 << 0)) pMemMap[0     + i] = Mem + ((i - cStart) << 8); // READ
		if (nFlags & (1 << 1)) pMemMap[0x100 + i] = Mem + ((i - cStart) << 8); // WRITE
		if (nFlags & (1 << 2)) pMemMap[0x200 + i] = Mem + ((i - cStart) << 8); // OP
		if (nFlags & (1 << 3)) pMemMap[0x300 + i] = Mem + ((i - cStart) << 8); // ARG
	}
}

INT32 Zet2MapArea(INT32 nStart, INT32 nEnd, INT32 nMode, UINT8 *Mem)
{
	UINT8 cStart = (nStart >> 8);
	UINT8 **pMemMap = Zet2CPUContext[nOpenedCPU]->pZet2MemMap;

	for (UINT16 i = cStart; i <= (nEnd >> 8); i++) {
		switch (nMode) {
			case 0: {
				pMemMap[0     + i] = Mem + ((i - cStart) << 8);
				break;
			}
		
			case 1: {
				pMemMap[0x100 + i] = Mem + ((i - cStart) << 8);
				break;
			}
			
			case 2: {
				pMemMap[0x200 + i] = Mem + ((i - cStart) << 8);
				pMemMap[0x300 + i] = Mem + ((i - cStart) << 8);
				break;
			}
		}
	}

	return 0;
}

INT32 Zet2MapArea(INT32 nStart, INT32 nEnd, INT32 nMode, UINT8 *Mem01, UINT8 *Mem02)
{

	UINT8 cStart = (nStart >> 8);
	UINT8 **pMemMap = Zet2CPUContext[nOpenedCPU]->pZet2MemMap;
	
	if (nMode != 2) {
		return 1;
	}
	
	for (UINT16 i = cStart; i <= (nEnd >> 8); i++) {
		pMemMap[0x200 + i] = Mem01 + ((i - cStart) << 8);
		pMemMap[0x300 + i] = Mem02 + ((i - cStart) << 8);
	}

	return 0;
}

void Zet2Reset()
{
	Z80Reset();
}

UINT32 Zet2GetPC(INT32 n)
{

	if (n < 0) {
		return ActiveZ80GetPC();
	} else {
		return Zet2CPUContext[n]->reg.pc.w.l;
	}
}

INT32 Zet2GetPrevPC(INT32 n)
{


	if (n < 0) {
		return ActiveZ80GetPrevPC();
	} else {
		return Zet2CPUContext[n]->reg.prvpc.d;
	}
}

INT32 Zet2Bc(INT32 n)
{

	if (n < 0) {
		return ActiveZ80GetBC();
	} else {
		return Zet2CPUContext[n]->reg.bc.w.l;
	}
}

INT32 Zet2De(INT32 n)
{
	if (n < 0) {
		return ActiveZ80GetDE();
	} else {
		return Zet2CPUContext[n]->reg.de.w.l;
	}
}

INT32 Zet2HL(INT32 n)
{

	if (n < 0) {
		return ActiveZ80GetHL();
	} else {
		return Zet2CPUContext[n]->reg.hl.w.l;
	}
}

INT32 Zet2I(INT32 n)
{

	if (n < 0) {
		return ActiveZ80GetI();
	} else {
		return Zet2CPUContext[n]->reg.i;
	}
}

INT32 Zet2Scan(INT32 nAction)
{

	if ((nAction & ACB_DRIVER_DATA) == 0) {
		return 0;
	}

	char szText[] = "Z80 #0";
	
	for (INT32 i = 0; i < nCPUCount; i++) {
		szText[5] = '1' + i;

		ScanVar(&Zet2CPUContext[i]->reg, sizeof(Z80_Regs), szText);
		SCAN_VAR(Z80EA[i]);
		SCAN_VAR(nZ80ICount[i]);
		SCAN_VAR(nZet2CyclesDone[i]);
	}
	
	SCAN_VAR(nZet2CyclesTotal);	

	return 0;
}

void Zet2SetIRQLine(const INT32 line, const INT32 status)
{

	switch ( status ) {
		case ZET2_IRQSTATUS_NONE:
			Z80SetIrqLine(line, 0);
			break;
		case ZET2_IRQSTATUS_ACK: 	
			Z80SetIrqLine(line, 1);
			break;
		case ZET2_IRQSTATUS_AUTO:
			Z80SetIrqLine(line, 1);
			Z80Execute(0);
			Z80SetIrqLine(0, 0);
			Z80Execute(0);
			break;
		case ZET2_IRQSTATUS_HOLD:
			Z80SetIrqLine(line, 1);
			Z80Execute(100);
			Z80SetIrqLine(0, 0);
			Z80Execute(0);
			break;
		case ZET2_IRQSTATUS_HOLD2:
			ActiveZ80SetIRQHold();
			Z80SetIrqLine(line, 1);
			break;
	}
}

void Zet2SetVector(INT32 vector)
{
	Z80Vector = vector;
}

UINT8 Zet2GetVector()
{
	return Z80Vector;
}

INT32 Zet2Nmi()
{
	Z80SetIrqLine(Z80_INPUT_LINE_NMI, 1);
	Z80Execute(0);
	Z80SetIrqLine(Z80_INPUT_LINE_NMI, 0);
	Z80Execute(0);
	INT32 nCycles = 12;
	nZet2CyclesTotal += nCycles;

	return nCycles;
}

INT32 Zet2Idle(INT32 nCycles)
{
	nZet2CyclesTotal += nCycles;

	return nCycles;
}

INT32 Zet2SegmentCycles()
{
	return z80TotalCycles();
}

INT32 Zet2TotalCycles()
{
	return nZet2CyclesTotal + z80TotalCycles();
}

void Zet2SetBUSREQLine(INT32 nStatus)
{

	if (nOpenedCPU < 0) return;
	
	Zet2CPUContext[nOpenedCPU]->BusReq = nStatus;
}

void Zet2SetHL(INT32 n, UINT16 value)
{
	Zet2CPUContext[n]->reg.hl.w.l=value;
}

void Zet2SetSP(INT32 n, UINT16 value)
{
	Zet2CPUContext[n]->reg.sp.w.l=value;
}

#undef MAX_Z80
